/* Interface to byteswapping functions.
   Copyright (C) 2006-2019 Free Software Foundation, Inc.

   This file is part of libctf.

   libctf is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 3, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
   See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.  If not see
   <http://www.gnu.org/licenses/>.  */

#ifndef _CTF_SWAP_H
#define _CTF_SWAP_H

#include "config.h"
#include <stdint.h>

#ifdef HAVE_BYTESWAP_H
#include <byteswap.h>
#else

#ifndef __DJGPP__
/* Provide our own versions of the byteswap functions.  */
inline uint16_t
bswap_16 (uint16_t v)
{
  return ((v >> 8) & 0xff) | ((v & 0xff) << 8);
}

inline uint32_t
bswap_32 (uint32_t v)
{
  return (  ((v & 0xff000000) >> 24)
	  | ((v & 0x00ff0000) >>  8)
	  | ((v & 0x0000ff00) <<  8)
	  | ((v & 0x000000ff) << 24));
}

inline uint64_t
bswap_identity_64 (uint64_t v)
{
  return v;
}

inline uint64_t
bswap_64 (uint64_t v)
{
  return (  ((v & 0xff00000000000000ULL) >> 56)
	  | ((v & 0x00ff000000000000ULL) >> 40)
	  | ((v & 0x0000ff0000000000ULL) >> 24)
	  | ((v & 0x000000ff00000000ULL) >>  8)
	  | ((v & 0x00000000ff000000ULL) <<  8)
	  | ((v & 0x0000000000ff0000ULL) << 24)
	  | ((v & 0x000000000000ff00ULL) << 40)
	  | ((v & 0x00000000000000ffULL) << 56));
}
#else  /* __DJGPP__ */

/*  The inline functions make trouble with diferent versions of gcc thus use macros.  */
# if defined (__GNUC__) && (__GNUC__ > 2 || (__GNUC__ == 2 && __GNUC_MINOR__ >= 8))
#  define __gnuc_extension__  __extension__
# else
#  define __gnuc_extension__
# endif

# define bswap_identity_64(v)                                      \
  (__gnuc_extension__                                              \
    ({                                                             \
       uint64_t value = (v);                                       \
       value;                                                      \
    })                                                             \
  )

# define bswap_16(v)                                               \
  (__gnuc_extension__                                              \
    ({                                                             \
       uint16_t _v = (v);                                          \
       uint16_t value = (  ((_v >> 8) & 0xFF)                      \
                         | ((_v & 0xFF) << 8));                    \
       value;                                                      \
    })                                                             \
  )

# define bswap_32(v)                                               \
  (__gnuc_extension__                                              \
    ({                                                             \
       uint32_t _v = (v);                                          \
       uint32_t value = (  ((_v & 0xFF000000) >> 24)               \
                         | ((_v & 0x00FF0000) >>  8)               \
                         | ((_v & 0x0000FF00) <<  8)               \
                         | ((_v & 0x000000FF) << 24));             \
       value;                                                      \
    })                                                             \
  )

# define bswap_64(v)                                               \
  (__gnuc_extension__                                              \
    ({                                                             \
       uint64_t _v = (v);                                          \
       uint64_t value = (  ((_v & 0xFF00000000000000ULL) >> 56)    \
                         | ((_v & 0x00FF000000000000ULL) >> 40)    \
                         | ((_v & 0x0000FF0000000000ULL) >> 24)    \
                         | ((_v & 0x000000FF00000000ULL) >>  8)    \
                         | ((_v & 0x00000000FF000000ULL) <<  8)    \
                         | ((_v & 0x0000000000FF0000ULL) << 24)    \
                         | ((_v & 0x000000000000FF00ULL) << 40)    \
                         | ((_v & 0x00000000000000FFULL) << 56));  \
       value;                                                      \
    })                                                             \
  )
#endif /* __DJGPP__ */

#endif /* !defined(HAVE_BYTESWAP_H) */

#endif /* !defined(_CTF_SWAP_H) */
