#include <ctype.h>
#include <errno.h>
#include <setjmp.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <io.h>

#include "config.h"
#include "command.h"
#include "general.h"
#include "error.h"
#include "variables.h"
#include "subst.h"
#include "externs.h"
#include "builtins/common.h"
#include "dosutil.h"
#include "hashcmd.h"

static char *encode_drivename_inplace (char *);
static char *add_sysroot_internal (char *);

#define CURRENT_PID ((pid_t) 1)
#if defined (__MSDOS__)
#define SYSROOT_COMPARE strnicmp
#define ROOTDEV_COMPARE strnicmp
#else
#define SYSROOT_COMPARE strncmp
#define ROOTDEV_COMPARE strncmp
#endif
#define SAVESTRING(p)  ((p != NULL) ? savestring (p) : NULL)
#define FREE_AND_CLEAR(p)  (free (p), p = NULL)

int dosutil_path_expand = 0;
int dosutil_path_separator = ';';
int dosutil_path_slash = '/';
char dosutil_sysroot[PATH_MAX + 1] = { '\0' };
int dosutil_sysroot_len = -1;
int dosutil_wait_status = -1;
pid_t dosutil_current_pid = CURRENT_PID;
const char dev_str[] = "/dev/";
const int dev_str_len = sizeof(dev_str) - 1;
const int dosutil_encoded_drive_len_diff = 4;
char dosutil_tmpdir[FILENAME_MAX + 1];
size_t dosutil_tmpdir_len;
int dosutil_test_finds_exe = 0;

static char *need_fix_variables[] =
{
  "PWD",
  "OLDPWD",
  "HOME",
  "PATH",
  "CDPATH",
  "COMSPEC",
  "MAILPATH",
  NULL,
};

/* Translate filenames of the form 'x:/dir/file' to '/dev/x/dir1/file'.
   Returns a new string. */

char *
dosutil_encode_drivename (char *path)
{
  int len = strlen (path);
  char *enc_path = (char *)malloc (len + dosutil_encoded_drive_len_diff);

  strcpy (enc_path, dev_str);
  enc_path[5] = path[0];
  strcpy (enc_path + sizeof (dev_str), path + 2);

  free (path);
  return enc_path;
}

/* Translate filenames of the form 'x:/dir/file' to '/dev/x/dir1/file'.
   Returns the same string that was passed in. This function assumes
   there is enough space to handle the modified string. */

static char *
encode_drivename_inplace (char *path)
{
  int len = strlen (path);
  char drive = path[0];

  memmove (path + 6, path + 2, len - 1);
  memcpy (path, dev_str, sizeof(dev_str) - 1);
  path[5] = drive;

  return path;
}

/* Translates filenames of the form '/dev/x/dir/file' into 'x:/dir/file'. */

char *
dosutil_decode_drivename (char *path)
{
  char *p = path;
  p = (path && path[1] == ':') ? (path + 2) : path;
  path[0] = p[5];
  path[1] = ':';
  strcpy (path + 2, p + 6);
  return path;
}

/* Returns whether or not a path starts with '/dev/' */

int
dosutil_is_dev_path (const char *path)
{
  if (path && path[1] == ':')
    path += 2;

  return ROOTDEV_COMPARE (dev_str, path, dev_str_len) == 0;
}

/* Returns whether or not a filename is in the form '/dev/x/dir1/file'. */

int
dosutil_is_encoded_drivename (const char *path)
{
  if (path && path[1] == ':')
    path += 2;

  return (ROOTDEV_COMPARE (dev_str, path, dev_str_len) == 0
      && path[dev_str_len]
      && (path[dev_str_len + 1] == '/' || path[dev_str_len + 1] == '\0'));
}

/* Returns whether the pathname is the root directory.  */

int
dosutil_is_root_directory_pathname (const char *path)
{
  const char *p = (path && path[1] == ':') ? (path + 2) : path;

  if (*p == '/' && p[1] == '\0')
    return 1;

  if (dosutil_is_encoded_drivename (path))
    return (p[6] == '/' && p[7] == '\0');
}
    
#define DOSUTIL_VERSION "r0.6"
char *
dosutil_version_string (void)
{
  static char version_string[128];

  sprintf (version_string, "%s", DOSUTIL_VERSION);
#if defined (USE_MULTIBYTE)
  strcat (version_string, " w/multibyte extension");
#endif

  return version_string;
}

char *
dosutil_make_import_path (char *value)
{
  char *p, *newenv;
  int newlen, pos;

  if (value == NULL)
    return NULL;

  /* fix each element */
  newlen = strlen (value) + 1;
  newenv = malloc (newlen);
  if (newenv != NULL)
    {
      char *to, *from;

      /* copy to buffer */
      from = value;
      to = newenv;

      while (*from)
      {
        if (*from != '\\')
          *to = *from;
        else
          *to = '/';
        ++to;
        ++from;
      }
      *to = '\0';

      /* may delete SYSROOT */
      for (p = newenv; p != NULL && *p; p = strchr (p + 1, ';'))
	{
	  int drive_len = 0;

	  /* replace delimiter */
	  while (*p == ';')
	    *p++ = dosutil_path_separator;
	  if (*p == '\0')
	    break;

	  /* check drive name */
          if (p[0] && p[1] == ':')
	    {
	      if (dosutil_sysroot_len > 0 && isalpha (dosutil_sysroot[0])
		  && dosutil_sysroot[1] == ':'
		  && tolower (p[0]) == dosutil_sysroot[0])
		drive_len = 2;
	      else if (dosutil_path_separator == ':')
		{
                  newlen += dosutil_encoded_drive_len_diff;
		  pos = p - newenv;
		  newenv = realloc (newenv, newlen);
		  if (newenv == NULL)
		    return NULL;
		  p = newenv + pos;
                  encode_drivename_inplace (p);
		}
	    }

	  /* delete `SYSROOT' */
	  if (dosutil_sysroot_len > 0
	      && SYSROOT_COMPARE (dosutil_sysroot + drive_len, p + drive_len,
				  dosutil_sysroot_len - drive_len) == 0)
	    {
	      strcpy (p, p + dosutil_sysroot_len);
	      newlen -= dosutil_sysroot_len;
	    }
	}

      /* may need to convert slash to back-slash */
      if (dosutil_path_slash == '\\')
	dosutil_tobslash (newenv);
    }

  return newenv;
}

char *
dosutil_make_export_path (char *value)
{
  char *p, *newenv;
  int newlen, pos;

  if (value == NULL)
    return NULL;

  /* fix each element */
  newlen = strlen (value) + 1;
  newenv = malloc (newlen);
  if (newenv != NULL)
    {
      /* copy to buffer */
      strcpy (newenv, value);
      /* convert back-slash to slash */
      dosutil_toslash (newenv);

      /* may add SYSROOT */
      for (p = newenv; p != NULL && *p; p = strchr (p + 1, dosutil_path_separator))
	{
	  /* replace delimiter */
	  while (*p == dosutil_path_separator)
	    *p++ = ';';
	  if (*p == '\0')
	    break;

            if (p[1] != dosutil_path_separator
                   && dosutil_is_encoded_drivename(p))
	    {
              /* Expand drive name */
              dosutil_decode_drivename (p);
              newlen -= dosutil_encoded_drive_len_diff;
	      pos = p - newenv;
	      newenv = realloc (newenv, newlen);
	      if (newenv == NULL)
		return NULL;
	      p = newenv + pos;
              /* Skip past colon after drive letter. */
	      p += 2;
	    }
          else if (dosutil_sysroot_len > 0 && p[0] == '/'
                   && !dosutil_is_dev_path (p))
	    {
	      /* expand SYSROOT */
	      pos = p - newenv;
	      newenv = realloc (newenv, newlen + dosutil_sysroot_len);
	      if (newenv == NULL)
		return NULL;
	      p = newenv + pos;
	      memmove (p + dosutil_sysroot_len, p, newlen - (p - newenv));
	      newlen += dosutil_sysroot_len;
	      memcpy (p, dosutil_sysroot, dosutil_sysroot_len);
	      p += dosutil_sysroot_len;
	    }
	}

      /* convert slash to back-slash */
      dosutil_tobslash (newenv);
    }

  return newenv;
}

void
dosutil_import_environment_variables (void)
{
  char *p;
  int i;

  p = getenv ("PATH_EXPAND");
  if (p != NULL)
    {
      if (*p == 'y' || *p == 'Y')
	dosutil_path_expand = 1;
    }
  p = getenv ("PATH_SEPARATOR");
  if (p != NULL)
  {
      if (*p != '\0' && *p != '/' && *p != '\\' && *p < 0x7f)
	dosutil_path_separator = (int) *(unsigned char *) p;
  }
  else
  {
    char buf[2];

    buf[0] = dosutil_path_separator;
    buf[1] = '\0';
    setenv("PATH_SEPARATOR", buf, 1);
  }

  p = getenv ("PATH_SLASH");
  if (p != NULL)
    {
      if (*p != '\0' && (*p == '/' || *p == '\\') && *p < 0x7f)
	dosutil_path_slash = (int) *(unsigned char *) p;
    }

  p = getenv (DOSUTIL_TEST_VAR_NAME);
  if (p != NULL)
    {
      if (*p == 'y' || *p =='Y')
        dosutil_test_finds_exe = 1;
    }

  p = getenv ("SYSROOT");
  if (p != NULL && *p != '\0')
    {
      if (p[1] == ':' && p[2] == '\0')
        strcpy(dosutil_sysroot, p);
      else
        _fixpath(p, dosutil_sysroot);

      if (isalpha (dosutil_sysroot[0]) && dosutil_sysroot[1] == ':')
        dosutil_sysroot[0] = tolower (dosutil_sysroot[0]);
      dosutil_sysroot_len = strlen (dosutil_sysroot);
      if (dosutil_sysroot[dosutil_sysroot_len - 1] == '/')
        dosutil_sysroot[--dosutil_sysroot_len] = '\0';

      setenv("SYSROOT", dosutil_sysroot, 1);
    }

#if 0
  for (i = 0; (p = need_fix_variables[i]) != NULL; i++)
    {
      char *ep, *fixp, *newbuf;

      ep = getenv (p);
      if (ep != NULL)
	{
	  fixp = dosutil_make_import_path (ep);
	  if (fixp != NULL)
	    {
	      newbuf = malloc (strlen (fixp) + strlen (p) + 2);
	      if (newbuf != NULL)
		{
		  sprintf (newbuf, "%s=%s", p, fixp);
		  putenv (newbuf);
		  free (newbuf);
		}
	      free (fixp);
	    }
	}
    }
#endif
}

int
dosutil_import_unixy_pathvar (char *name, char **path)
{
  int i;
  char *p;

  for (i = 0; (p = need_fix_variables[i]) != NULL; i++)
    {
      if (strcmp (p, name) == 0)
        break;
    }

  if (p == NULL)
    return 0;

  *path = dosutil_make_import_path (*path);
  return 1;
}

char *
dosutil_add_slash (char *path)
{
  char *top = path;

  path += strlen (path);
  if (path != top && path[-1] != '/')
    {
      *path++ = '/';
      *path++ = '\0';
    }

  return top;
}

char *
dosutil_delete_slash (char *path)
{
  char *top = path;

  path += strlen (path);
  if (path != top && path[-1] == '/')
    path[-1] = '\0';

  return top;
}

char *
dosutil_toslash (char *path)
{
  char *p;
#if defined (USE_MULTIBYTE)
  int mbsize;
#endif

  for (p = path; *p; p++)
    {
#if defined (USE_MULTIBYTE)
      mbsize = mblen (p, MB_CUR_MAX);
      if (mbsize > 1)
	{
	  p += mbsize - 1;
	  continue;
	}
      else if (*p == '\\')
	*p = '/';
#else
      if (*p == '\\')
	*p = '/';
#endif
    }

  return path;
}

char *
dosutil_tobslash (char *path)
{
  char *p;
#if defined (USE_MULTIBYTE)
  int mbsize;
#endif

  for (p = path; *p; p++)
    {
#if defined (USE_MULTIBYTE)
      mbsize = mblen (p, MB_CUR_MAX);
      if (mbsize > 1)
	{
	  p += mbsize - 1;
	  continue;
	}
      else if (*p == '/')
	*p = '\\';
#else
      if (*p == '/')
	*p = '\\';
#endif
    }

  return path;
}

char *
dosutil_tolower (char *path)
{
  char *p;
#if defined (USE_MULTIBYTE)
  int mbsize;
#endif

  for (p = path; *p; p++)
    {
#if defined (USE_MULTIBYTE)
      mbsize = mblen (p, MB_CUR_MAX);
      if (mbsize > 1)
	{
	  p += mbsize - 1;
	  continue;
	}
      else if (isupper (*p))
	*p = tolower (*p);
#else
      if (isupper (*p))
	*p = tolower (*p);
#endif
    }

  return path;
}

char *
dosutil_toupper (char *path)
{
  char *p;
#if defined (USE_MULTIBYTE)
  int mbsize;
#endif

  for (p = path; *p; p++)
    {
#if defined (USE_MULTIBYTE)
      mbsize = mblen (p, MB_CUR_MAX);
      if (mbsize > 1)
	{
	  p += mbsize - 1;
	  continue;
	}
      else if (islower (*p))
	*p = toupper (*p);
#else
      if (islower (*p))
	*p = toupper (*p);
#endif
    }

  return path;
}

static char *
add_sysroot_internal (char *path)
{
  if (path[0] == '/' && dosutil_sysroot_len > 0)
    {
      /* expand SYSROOT */
      char *p = malloc (strlen (path) + dosutil_sysroot_len + 2);
      if (p != NULL)
	{
	  strcpy (p, dosutil_sysroot);
	  strcpy (p + dosutil_sysroot_len, path);
	  free (path);
	  path = p;
	}
    }
  return path;
}

char *
dosutil_add_sysroot (char *path)
{
  char *p;

  dosutil_toslash (path);

  if (ROOTDEV_COMPARE (dev_str, path, dev_str_len) == 0)
      return path;

  return add_sysroot_internal (path);
}

char *
dosutil_delete_sysroot (char *path)
{
  int drive_len = 0;

  /* convert back-slash to slash */
  dosutil_toslash (path);

  if (dosutil_sysroot_len > 0)
    {
      /* check drive name */
      if (isalpha (path[0]) && path[1] == ':'
	  && isalpha (dosutil_sysroot[0]) && dosutil_sysroot[1] == ':')
	{
	  if (tolower (path[0]) != dosutil_sysroot[0])
	    return path;
	  drive_len = 2;
	}

      /* delete `SYSROOT' */
      if (SYSROOT_COMPARE (dosutil_sysroot + drive_len, path + drive_len,
			   dosutil_sysroot_len - drive_len) == 0)
        if (*(path + dosutil_sysroot_len) != '\0')
          strcpy (path, path + dosutil_sysroot_len);
        else
        {
          path[0] = '/';
          path[1] = '\0';
        }
    }

  return path;
}

char *
dosutil_expand_path (char *path)
{
  char *p;
  int is_dev = 0;

  dosutil_toslash (path);

  p = ((path && path[1] == ':') ? (path + 2) : path);
  is_dev = (ROOTDEV_COMPARE (dev_str, p, dev_str_len) == 0);

  if (is_dev)
  {
    if (dosutil_path_expand == 0)
      return path;
    if (p[dev_str_len] &&
         (p[dev_str_len + 1] == '/' || path[dev_str_len + 1] == '\0'))
      return dosutil_decode_drivename (path);
    else if (strncmp (p + dev_str_len, "env/", sizeof("env/") - 1) == 0)
    {
      /* In DJGPP 2.03, we could use _put_path and its return value.
         But in DJGPP 2.02, _put_path returns nothing.  */
      char buffer[FILENAME_MAX];
      _fixpath (path, buffer);
      free (path);
      path = malloc (strlen (buffer) + 1);
      strcpy (path, buffer);
    }
    else
      return path;
  }

  if (path[0] == '/' && dosutil_sysroot_len > 0)
    return add_sysroot_internal (path);

  return path;
}

char *
dosutil_fix_variable (char *vp)
{
  char *p, *fixp, *newbuf;
  int i;

  if (vp == NULL)
    return NULL;

  p = strchr (vp, '=');
  if (!p)
    return vp;

  *p = '\0';
  fixp = dosutil_make_export_path (p + 1);
  if (fixp != NULL)
  {
     newbuf = malloc (strlen (fixp) + strlen (vp) + 2);
     if (newbuf != NULL)
     {
#if defined (__MSDOS__) && 0
        /* convert upper case */
        if (strcmp (p, "PATH") == 0)
          dosutil_toupper (fixp);
#endif
        sprintf (newbuf, "%s=%s", vp, fixp);
        free (vp);
        vp = newbuf;
      }
    free (fixp);
  }

  return vp;
}

char **
dosutil_get_import_variables (void)
{
  char **p;
  int i;

  /* count need size */
  for (i = 0; need_fix_variables[i] != NULL; i++)
    ;
  p = malloc (i * sizeof (char*));
  if (p != NULL)
    {
      for (i = 0; need_fix_variables[i] != NULL; i++)
	p[i] = dosutil_make_export_path (get_string_value (need_fix_variables[i]));
    }

  return p;
}

void
dosutil_put_export_variables (char **vp)
{
  SHELL_VAR *var;
  int i;

  if (vp == NULL)
    return;

  for (i = 0; need_fix_variables[i] != NULL; i++)
    {
      var = find_variable (need_fix_variables[i]);
      if (var != NULL)
	{
	  if (var->value != NULL)
	    free (var->value);
	  var->value = dosutil_make_import_path (vp[i]);
	}
      if (vp[i] != NULL)
	free (vp[i]);
    }
  free (vp);
  set_working_directory (get_string_value ("PWD"));
}

void dosutil_set_tmpdir(const char *bashpath)
{
  char *tmpdirs[] = {"$TMPDIR", "$TEMP", "$TMP", "c:/tmp", "c:/", NULL};
  char *tmp, *dir;

#if 0
  char *tmpf;
#endif
  int fd, i, len;

  dosutil_tmpdir[0] = '\0';
  dosutil_tmpdir_len = 0;

  i = 0;
  while (tmpdirs[i] != NULL)
  {
    if (tmpdirs[i][0] == '$')
    {
      dir = getenv(tmpdirs[i] + 1);
      if (!(dir && *dir))
      {
        ++i;
        continue;
      }
    }
    else
    {
      dir = tmpdirs[i];
    }


    if (*dir && dir[1] == ':' && dir[2] == '\0')
    {
      dir[2] = '/';
      dir[3] = '\0';
    }

    if (access(dir, D_OK) >= 0)
      break;

    ++i;
  }

  if (tmpdirs[i] == NULL)
    return;

#if 0
  /* Now check and see if the directory is writeable. */
  tmpf = tempnam(dir, "dj");
  fd = open (tmpf, O_RDWR | O_CREAT | O_TRUNC, 0666);
  if (fd < 0)
    continue;
  close(fd);
  remove(tmpf);
#endif

  strcpy(dosutil_tmpdir, dir);
  dosutil_toslash(dosutil_tmpdir);
  dosutil_tmpdir_len = strlen(dosutil_tmpdir);
  if (dosutil_tmpdir[dosutil_tmpdir_len - 1] == '/')
  {
    dosutil_tmpdir[dosutil_tmpdir_len - 1] = '\0';
    --dosutil_tmpdir_len;
  }

  setenv("TMPDIR", dosutil_tmpdir, 1);
}

#if defined (__DJGPP__)
#define _DEV_STDIN  0x0001
#define _DEV_STDOUT 0x0002
#define _DEV_NUL    0x0004
#define _DEV_CLOCK  0x0008
#define _DEV_RAW    0x0020
#define _DEV_CDEV   0x0080
#define _DEV_IOCTRL 0x4000

void
dosutil_reserve_fds (void)
{
  int fd;
  short devmod;

  /* don't use fd 5 and 6. these are used by configure script. */
  if ((devmod = _get_dev_info (5)) == -1)
    {
      fd = open ("NUL", O_RDWR);
      if (fd != 5)
	{
	  dup2 (fd, 5);
	  close (fd);
	}
    }

  if ((devmod = _get_dev_info (6)) == -1)
    {
      fd = open ("NUL", O_RDWR);
      if (fd != 6)
	{
	  dup2 (fd, 6);
	  close (fd);
	}
    }
}
#else
void
dosutil_reserve_fds (void)
{
}
#endif

#if defined (__DJGPP__)
static char dosutil_oldcwd[PATH_MAX + 1] = { '\0' };

static void
dosutil_restore_cwd (void)
{
  if (dosutil_oldcwd[0] != '\0')
    (void) (chdir) (dosutil_oldcwd);
}

void
dosutil_save_cwd (void)
{
  (void) (getcwd) (dosutil_oldcwd, sizeof (dosutil_oldcwd));
  atexit (dosutil_restore_cwd);
}
#else
void
dosutil_save_cwd (void)
{
}
#endif

void
dosutil_save_std_fds (int fds[3])
{
  int i;

  /* save stdin/out/err */
  for (i = 0; i < 3; i++)
#if defined (__DJGPP__)
    if ((fds[i] = fcntl (i, F_DUPFD, 20)) < 0)
#else
    if ((fds[i] = dup (i)) < 0)
#endif
      internal_error ("Cannot duplicate fd %d: %s",
		      i, strerror (errno));
}

void
dosutil_restore_std_fds (int fds[3])
{
  int i;

  /* restore stdin/out/err */
  for (i = 0; i < 3; i++)
    if (fds[i] >= 0)
      {
#if defined (__DJGPP__)
	if (dosutil_dup2_wrapper (fds[i], i) < 0)
#else
	if (dup2 (fds[i], i) < 0)
#endif
	  internal_error ("cannot duplicate fd %d to fd %d: %s",
			  fds[i], i, strerror (errno));
	close (fds[i]);
	fds[i] = -1;
      }
}

static void
save_jmp_buf (jmp_buf save, jmp_buf now)
{
  memcpy ((char *) save, (char *) now, sizeof (jmp_buf));
}

static void
restore_jmp_buf (jmp_buf save, jmp_buf now)
{
  memcpy ((char *) now, (char *) save, sizeof (jmp_buf));
}

extern jmp_buf top_level, subshell_top_level;
extern int interactive, interactive_shell, login_shell;
extern int subshell_environment;
extern int subshell_exit_builtin, exit_immediately_on_error;
extern int dollar_dollar_pid;
extern int array_needs_making;
extern char **temporary_env, **function_env, **builtin_env;

static void
save_current_directory (OLDENVBUF *envp)
{
  envp->pwd    = SAVESTRING (get_string_value ("PWD"));
#if 0
  envp->oldpwd = SAVESTRING (get_string_value ("OLDPWD"));
#endif
  envp->curdir = dosutil_getcwd (NULL, PATH_MAX);
}

static void
restore_current_directory (OLDENVBUF *envp)
{
  /* change old directory */
  if (envp->curdir != NULL)
    {
      dosutil_chdir (envp->curdir);
      if (envp->pwd != NULL)
	set_working_directory (envp->pwd);
      else
	set_working_directory (envp->curdir);
      FREE_AND_CLEAR (envp->curdir);
    }
  else if (envp->pwd != NULL)
    {
      dosutil_chdir (envp->pwd);
      set_working_directory (envp->pwd);
    }

#if 0
  /* replace variables */
  unbind_variable ("PWD");
  if (envp->pwd)
    {
      bind_variable ("PWD", envp->pwd);
      FREE_AND_CLEAR (envp->pwd);
    }

  unbind_variable ("OLDPWD");
  if (envp->oldpwd)
    {
      bind_variable ("OLDPWD", envp->oldpwd);
      FREE_AND_CLEAR (envp->oldpwd);
    }
#else
  if (envp->pwd)
    FREE_AND_CLEAR (envp->pwd);
#endif
}

static void
save_global_variables (OLDENVBUF *envp)
{
  envp->interactive               = interactive;
  envp->interactive_shell         = interactive_shell;
  envp->login_shell               = login_shell;
  envp->subshell_environment      = subshell_environment;
  envp->subshell_exit_builtin     = subshell_exit_builtin;
  envp->exit_immediately_on_error = exit_immediately_on_error;
  envp->variable_context          = variable_context;
  envp->dollar_dollar_pid         = dollar_dollar_pid;
}

static void
restore_global_variables (OLDENVBUF *envp)
{
  interactive               = envp->interactive;
  interactive_shell         = envp->interactive_shell;
  login_shell               = envp->login_shell;
  subshell_environment      = envp->subshell_environment;
  subshell_exit_builtin     = envp->subshell_exit_builtin;
  exit_immediately_on_error = envp->exit_immediately_on_error;
  variable_context          = envp->variable_context;
  dollar_dollar_pid         = envp->dollar_dollar_pid;
}

static HASH_TABLE *
copy_shell_variables_hash_table (HASH_TABLE *table)
{
  int i;
  HASH_TABLE *new_table;
  BUCKET_CONTENTS *new_item, *old_item;
  SHELL_VAR *new_var, *old_var;

  if (!table)
    return (HASH_TABLE *) NULL;

  new_table = make_hash_table (table->nbuckets);
  new_table->nentries = table->nentries;
  for (i = 0; i < table->nbuckets; i++)
    {
      old_item = table->bucket_array[i];
      new_item = (BUCKET_CONTENTS *) NULL;
      while (old_item)
	{
	  if (new_item)
	    {
	      new_item->next = (BUCKET_CONTENTS *) xmalloc (sizeof (BUCKET_CONTENTS));
	      new_item = new_item->next;
	    }
	  else
	    {
	      new_table->bucket_array[i] =
		(BUCKET_CONTENTS *) xmalloc (sizeof (BUCKET_CONTENTS));
	      new_item = new_table->bucket_array[i];
	    }
	  new_item->data = (char *) copy_variable ((SHELL_VAR *) old_item->data);
	  old_var = (SHELL_VAR *)old_item->data;
	  new_var = (SHELL_VAR *)new_item->data;
	  while (old_var->prev_context)
	    {
	      new_var->prev_context = copy_variable (old_var->prev_context);
	      new_var = new_var->prev_context;
	      old_var = old_var->prev_context;
	    }
	  new_item->next = (BUCKET_CONTENTS *)NULL;
	  new_item->key = SAVESTRING (old_item->key);
	  new_item->times_found = old_item->times_found;
	  old_item = old_item->next;
	}
    }

  return new_table;
}

static void
dispose_shell_variables_hash_table (HASH_TABLE *table)
{
  int i;
  BUCKET_CONTENTS *item, *next;
  SHELL_VAR *var, *prev;

  if (!table)
    return;

  for (i = 0; i < table->nbuckets; i++)
    {
      item = table->bucket_array[i];
      while (item)
	{
	  var = (SHELL_VAR *) item->data;
	  while (var)
	    {
	      prev = var->prev_context;
	      dispose_variable (var);
	      var = prev;
	    }
	  if (item->key)
	    free (item->key);
	  next = item->next;
	  free (item);
	  item = next;
	}
    }
  free (table->bucket_array);
  free (table);
}

static void
save_shell_variables (OLDENVBUF *envp)
{
  maybe_make_export_env ();

  envp->shell_variables = shell_variables;
  shell_variables = copy_shell_variables_hash_table (shell_variables);
  envp->shell_functions = shell_functions;
  shell_functions = copy_shell_variables_hash_table (shell_functions);

  envp->temporary_env = temporary_env;
  if (temporary_env != NULL)
    temporary_env = copy_array (temporary_env);
  envp->function_env = function_env;
  if (function_env != NULL)
    function_env = copy_array (function_env);
  envp->builtin_env = builtin_env;
  if (builtin_env != NULL)
    builtin_env = copy_array (builtin_env);
#if 0
  /* made by maybe_make_export_env () */
  envp->export_env = export_env;
  if (export_env != NULL)
    export_env = copy_array (export_env);
#endif

  envp->rest_of_args = list_rest_of_args ();
}

static void
restore_shell_variables (OLDENVBUF *envp)
{
  dispose_shell_variables_hash_table (shell_variables);
  shell_variables = envp->shell_variables;
  dispose_shell_variables_hash_table (shell_functions);
  shell_functions = envp->shell_functions;

  free_array (temporary_env);
  temporary_env = envp->temporary_env;
  free_array (function_env);
  function_env = envp->function_env;
  free_array (builtin_env);
  builtin_env = envp->builtin_env;
#if 0
  /* made by maybe_make_export_env () */
  free_array (export_env);
  export_env = envp->export_env;
#endif

  remember_args (envp->rest_of_args, 1);

  array_needs_making = 1;
  maybe_make_export_env ();
}

int
dosutil_save_all_environment (OLDENVBUF *envp)
{
  save_jmp_buf (envp->top_level, top_level);
  save_jmp_buf (envp->subshell_top_level, subshell_top_level);
  dosutil_save_std_fds (envp->fds);
  save_current_directory (envp);
  save_global_variables (envp);
  save_shell_variables (envp);
  return 0;
}

int
dosutil_restore_all_environment (OLDENVBUF *envp)
{
  restore_shell_variables (envp);
  restore_global_variables (envp);
  restore_current_directory (envp);
  dosutil_restore_std_fds (envp->fds);
  restore_jmp_buf (envp->top_level, top_level);
  restore_jmp_buf (envp->subshell_top_level, subshell_top_level);
  return 0;
}

/* pipe emulation */

struct pipe_t
{
  int fds[2];
  char *name;
};

#if !defined (OPEN_MAX)
#define OPEN_MAX 256
#endif
static struct pipe_t pipes[OPEN_MAX];
static int pipes_max = 0;

void
dosutil_pipe_clean (int no)
{
  int i;

  for (i = no; i < pipes_max; i++)
    {
      remove (pipes[i].name);
      free (pipes[i].name);
      pipes[i].fds[0] = pipes[i].fds[1] = 0;
      pipes[i].name = NULL;
    }

  pipes_max = no;
}

int
dosutil_pipe_no (void)
{
  return pipes_max;
}

int
dosutil_pipe (int fds[2])
{
  int ifd;
  int ofd;
  char *tname;

  if (pipes_max >= OPEN_MAX)
    {
      errno = EMFILE;
      return -1;
    }

  tname = malloc (dosutil_tmpdir_len + sizeof ("/ppXXXXXX"));
  if (tname == NULL)
    return -1;
  sprintf (tname, "%s/ppXXXXXX", dosutil_tmpdir);
  tname = mktemp (dosutil_expand_path (tname));
  if (tname == NULL)
    return -1;

  ofd = open (tname, O_RDWR|O_CREAT|O_TRUNC, 0666);
  if (ofd < 0)
    {
      free (tname);
      return -1;
    }
#if defined (__DJGPP__)
  if (ofd < 20)
    {
      int tfd;

      tfd = fcntl (ofd, F_DUPFD, 20);
      close (ofd);
      if (tfd < 0)
	{
	  remove (tname);
	  free (tname);
	  return -1;
	}
      ofd = tfd;
    }
  ifd = fcntl (ofd, F_DUPFD, 20);
#else
  ifd = dup (ofd);
#endif
  if (ifd < 0)
    {
      close (ofd);
      remove (tname);
      free (tname);
      return -1;
    }

  pipes[pipes_max].name = tname;
  fds[0] = pipes[pipes_max].fds[0] = ifd;
  fds[1] = pipes[pipes_max].fds[1] = ofd;

  pipes_max++;

  return 0;
}

#include <sys/param.h>
#if !defined (MAXPID) 
#define MAXPID ((pid_t) 30000)
#endif

static pid_t last_pid = CURRENT_PID;

pid_t
dosutil_make_child_pid (void)
{
  if (++last_pid > MAXPID)
    last_pid = CURRENT_PID + 1;

  return last_pid;
}

/* `SYSROOT' extention functions */
int
dosutil_chdir (char *path)
{
  char *fixdirp;
  int rc;

  if (path[0] == '/' && dosutil_sysroot_len > 0
      && !dosutil_is_dev_path(path))
    {
      fixdirp = malloc (strlen (path) + dosutil_sysroot_len + 1);
      if (fixdirp != NULL)
	{
	  sprintf (fixdirp, "%s%s", dosutil_sysroot, path);
	  rc = (chdir) (fixdirp);
	  free (fixdirp);
	  return rc;
	}
    }

  return (chdir) (path);
}

char *
dosutil_getcwd (char *buffer, int length)
{
  char *p;
  int buflen = length;

  if (dosutil_path_separator == ':')
    if (buffer != NULL)
      buflen -= dosutil_encoded_drive_len_diff;
    else
      buflen += dosutil_encoded_drive_len_diff;

  p = (getcwd) (buffer, buflen);
  if (p != NULL)
    {
      dosutil_delete_sysroot (p);
      if (isalpha (p[0]) && p[1] == ':' && dosutil_path_separator == ':'
	  && strlen (p) < (length - 1))
        p = encode_drivename_inplace (p);
    }

  return p;
}

char *
dosutil_getwd (char *buffer)
{
  char *p;

  p = (getwd) (buffer);
  if (p != NULL)
    {
      dosutil_delete_sysroot (p);
      if (isalpha (p[0]) && p[1] == ':' && dosutil_path_separator == ':')
        p = dosutil_encode_drivename (p);
    }

  return p;
}

#include <dos.h>
#include <dpmi.h>
#include <go32.h>
#include <pc.h>
#include <sys/exceptn.h>

/* default stack size */
unsigned _stklen = 1024 * 1024;

static int dosutil_cbrk_vector = -1;

/* The list of executable suffixes that add_executable_suffix
   will look for. */

static char *executable_suffixes[] =
{
  "com",
  "exe",
  "bat",
  "btm",
  "sh",
  "ksh",
  "pl",
  "sed",
  "awk",
  NULL,
};

/* Returns the command string if a suffix was added,
   or NULL if one wasn't added. */ 

/* _USE_LFN is a macro in DJGPP's fcntl.h that reports if LFN is enabled. */

#ifndef _USE_LFN
/* Assume no LFNs when not using DJGPP. */
#define _USE_LFN 0
#endif

char *
dosutil_add_executable_suffix (char *command)
{
  int len, index;
  char *ext;
  struct stat finfo;
  int lfn = _USE_LFN;
  int has_dot = 0;
  char *path = command;
  int attr;
  /* If file already exists and it isn't a directory,
     don't bother checking for extensions. */
  if ((attr = _chmod(command, 0)) >= 0 && !(attr & 0x10))
#if 0
  if ((stat (command, &finfo) == 0) && !S_ISDIR(finfo.st_mode))
#endif
    return command;

  index = len = strlen(command);

  if (command[0] && (command[1] == ':'))
  {
    index -= 2;
    len -= 2;
    path += 2;
  }

  /* Find extension of command, if any. */
  while ((--index >= 0) && (path[index] != '/') && path[index] != '.')
  ;

  /* If a dot is the first character found and has characters following it,
     then an extension is already present and don't do the search. */

  has_dot = ( (index >= 0) && (path[index] == '.') );
  if ((has_dot) && (index+1 == len) && !lfn)
    return command;

  if (has_dot && !lfn)
    return command;
  else
  {
    ext = path + len;
    *ext++ = '.';
  }

  for (index = 0; executable_suffixes[index] != NULL; index++)
  {
    strcpy (ext, executable_suffixes[index]);
    if ((attr = _chmod(command, 0)) >= 0 && !(attr & 0x10))
#if 0
    if ((stat (command, &finfo) == 0) && !S_ISDIR(finfo.st_mode))
#endif
      return command;
  }

  /* If a period was added, erase it. */
  path[len] = '\0';
  return command;
}

int
dosutil_has_executable_suffix (const char *command)
{
  int len, index;
  const char *ext;
  int has_dot = 0;

  index = len = strlen(command);

/* Find extension of command, if any. */
  while ((--index >= 0) && (command[index] != '/') && (command[index] != '.'))
  ;

  /* Return false when no dot or no suffix after the dot. */
  has_dot = ( (index >= 0) && (command[index] == '.') );
  if (!has_dot || (has_dot && (index+1 == len)))
    return 0;

  ext = command + index + 1;

  for (index = 0; executable_suffixes[index] != NULL; index++)
  {
    /* If we find an executable extension, then return success. */
    if (stricmp (ext, executable_suffixes[index]) == 0)
      return 1;
  }

  /* Failed to find suffix in the executable suffix list. */
  return 0;
}

int
dosutil_has_suffix (const char *command)
{
  int len, index;
  const char *ext;
  int has_dot = 0;

  index = len = strlen(command);

  /* Find extension of command, if any. */
  while ((--index >= 0) && (command[index] != '/') && (command[index] != '.'))
  ;

  /* Return false when no dot. */
  has_dot = ( (index >= 0) && (command[index] == '.') );
  if (has_dot)
    return -1;
  else
    return len;
}

void
dosutil_save_exceptions (OLDEXCEPTBUF *exceptionp)
{
  int i;

  if (dosutil_cbrk_vector < 0)
    {
      if (ScreenPrimary != 0xa0000)
	dosutil_cbrk_vector = 0x1b;
      else
	dosutil_cbrk_vector = 0x06;
    }

  for (i = 0; i < DOSUTIL_EXCEPTION_COUNT; i++)
    __dpmi_get_processor_exception_handler_vector (i, &exceptionp->except_original[i]);
  __dpmi_get_protected_mode_interrupt_vector (9, &exceptionp->kbd_original);
  __dpmi_get_protected_mode_interrupt_vector (0x75, &exceptionp->npx_original);
  __dpmi_get_real_mode_interrupt_vector (dosutil_cbrk_vector, &exceptionp->cbrk_original);
}

void
dosutil_restore_exceptions (OLDEXCEPTBUF *exceptionp)
{
  int i;

  for (i = 0; i < DOSUTIL_EXCEPTION_COUNT; i++)
    __dpmi_set_processor_exception_handler_vector (i, &exceptionp->except_original[i]);
  __dpmi_set_protected_mode_interrupt_vector (9, &exceptionp->kbd_original);
  __dpmi_set_protected_mode_interrupt_vector (0x75, &exceptionp->npx_original);
  __dpmi_set_real_mode_interrupt_vector (dosutil_cbrk_vector, &exceptionp->cbrk_original);
}

#define CHECK_SELS 12
int *
dosutil_check_dpmi_selectors (void)
{
  int *p;
  int i;

  p = malloc (sizeof (int) * CHECK_SELS);
  if (p != NULL)
    {
      for (i = 0; i < CHECK_SELS; i++)
	p[i] = __dpmi_allocate_ldt_descriptors (1);
      for (i = 0; i < CHECK_SELS; i++)
	__dpmi_free_ldt_descriptor (p[i]);
    }

  return p;
}

void
dosutil_free_dpmi_selectors (int *selectors)
{
  int sel1;
  int i;

  if (selectors != NULL)
    {
      sel1 = __dpmi_allocate_ldt_descriptors (1);
      __dpmi_free_ldt_descriptor (sel1);
      if (sel1 != selectors[0])
	{
	  for (i = 0; i < CHECK_SELS; i++)
	    __dpmi_free_ldt_descriptor (selectors[i]);
	}
      free (selectors);
    }
}

int
dosutil_dup2_wrapper (int from, int to)
{
  int rc;
  int old_enable, old_ctrl_c;

  old_enable = __dpmi_get_and_disable_virtual_interrupt_state ();
  old_ctrl_c = __djgpp_set_ctrl_c (0);

  rc = dup2 (from, to);

  (void) __djgpp_set_ctrl_c (old_ctrl_c);
  if (old_enable)
    (void) __dpmi_get_and_enable_virtual_interrupt_state ();

  return rc;
}

char *
dosutil_make_response_file (char **argv)
{
  char *tname;

  tname = malloc (dosutil_tmpdir_len + sizeof ("/rsXXXXXX"));
  if (tname != NULL)
    {
      sprintf (tname, "%s/rsXXXXXX", dosutil_tmpdir);
      tname = mktemp (dosutil_expand_path (tname));
      if (tname != NULL)
	{
	  FILE *fp;
	  char *p;
	  int i;

	  fp = fopen (tname, "wt");
	  if (fp == NULL)
	    {
	      free (tname);
	      return NULL;
	    }

	  for (i = 1; (p = argv[i]) != NULL; i++)
	    {
	      if (i != 1)
		fputc (' ', fp);
	      if (strchr (p, '"') == NULL)
		{
		  fputc ('"', fp);
		  fputs (p, fp);
		  fputc ('"', fp);
		}
	      else
		{
		  fputc ('"', fp);
		  while (*p)
		    {
		      if (*p == '"')
			fputc ('\\', fp);
		      fputc (*p, fp);
		      p++;
		    }
		  fputc ('"', fp);
		}
	    }
	  fclose (fp);
	}
    }

  return tname;
}

#define _DEV_STDIN  0x0001
#define _DEV_STDOUT 0x0002
#define _DEV_NUL    0x0004
#define _DEV_CLOCK  0x0008
#define _DEV_RAW    0x0020
#define _DEV_EOF    0x0040
#define _DEV_CDEV   0x0080
#define _DEV_IOCTRL 0x4000

#define _REG_STATUS_CF 0x01
#define _REG_STATUS_ZF 0x40
void
dosutil_reset_console (void)
{
  unsigned short devinfo;
  int handle;
  __dpmi_regs r;

  handle = 0; /* STDIN (CON) */

  r.x.ax = 0x4400;
  r.x.bx = handle;
  __dpmi_int (0x21, &r);
  if (r.x.flags & _REG_STATUS_CF)
    return;
  devinfo = r.x.dx;

  if ((devinfo & _DEV_CDEV) && (devinfo & _DEV_STDIN) && (devinfo & _DEV_EOF) == 0)
    {
      r.x.ax = 0x4000; /* WRITE */
      r.x.bx = handle; /* STDIN (CON) */
      r.x.cx = 0; /* zero byte */
      r.x.dx = 0; /* dummy offset */
      r.x.ds = 0; /* dummy segment */
      __dpmi_int(0x21, &r);
      if (r.x.flags & _REG_STATUS_CF)
	return;
    }
}

static int delay_remove_size = 0;
static int delay_remove_counts = 0;
static char **delay_remove_filenames = NULL;
#define DELAY_REMOVE_DEFAULT_SIZE 5 /* at least size */

void
dosutil_add_delay_remove (const char *filename)
{
  int i;

  if (delay_remove_filenames == NULL)
    {
      delay_remove_size = DELAY_REMOVE_DEFAULT_SIZE;
      delay_remove_counts = 0;
      delay_remove_filenames = (char **) xmalloc (sizeof (char *)
						  * delay_remove_size);
      for (i = 0; i < delay_remove_size; i++)
	delay_remove_filenames[i] = NULL;
    }
  else if (delay_remove_size <= delay_remove_counts)
    {
      delay_remove_size *= 2;
      delay_remove_filenames = (char **) xrealloc (delay_remove_filenames,
						   sizeof (char *)
						   * delay_remove_size);
      for (i = delay_remove_counts; i < delay_remove_size; i++)
	delay_remove_filenames[i] = NULL;
    }

  for (i = 0; i < delay_remove_size; i++)
    if (delay_remove_filenames[i] == NULL)
      {
	delay_remove_filenames[i] = savestring (filename);
	delay_remove_counts++;
	break;
      }
}

void
dosutil_try_delay_remove (void)
{
  int i;

  if (delay_remove_counts == 0)
    return;

  /* search and delete if found */
  for (i = 0; i < delay_remove_size; i++)
    {
      if (delay_remove_filenames[i] == NULL)
	continue;

      if (remove (delay_remove_filenames[i]) == 0
          || (errno == ENOENT))
	{
	  free (delay_remove_filenames[i]);
	  delay_remove_filenames[i] = NULL;
	  delay_remove_counts--;
	}
    }

  if (delay_remove_counts == 0
      && delay_remove_size != DELAY_REMOVE_DEFAULT_SIZE)
    {
      /* shrink array size */
      delay_remove_size = DELAY_REMOVE_DEFAULT_SIZE;
      delay_remove_filenames = (char **) xrealloc (delay_remove_filenames,
						   sizeof (char *)
						   * delay_remove_size);
    }
  if (errno)
    errno = 0;
}

