/* A cheap pipe emulation for DOS.
   Uses one read/write file descriptor. */


#if defined (__DJGPP__)

#include <config.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <io.h>


struct pipe_t
{
  int rewind;
  int fds[3];
  char *name;
};

/*
 * RW_BIT:  0 = read end
 *          1 = write end
 */
#define RW_BIT     0x40000000

#define READ_END   0
#define WRITE_END  1
#define BACK_UP    2

#define IS_CLOSED(FD)      (pipes[i].fds[READ_END] == -1 && pipes[i].fds[WRITE_END] == -1 && pipes[i].fds[BACK_UP] == ((FD) & ~RW_BIT))
#define IS_NOT_CLOSED(FD)  (pipes[i].fds[BACK_UP] == ((FD) & ~RW_BIT) && !IS_CLOSED((FD)))
#define PIPE_FOUND(FD)     (pipes[i].fds[READ_END] == (FD) || pipes[i].fds[WRITE_END] == ((FD) | RW_BIT) || IS_CLOSED((FD)))

#if !defined (OPEN_MAX)
#define OPEN_MAX 256
#endif

static struct pipe_t pipes[OPEN_MAX];
static int pipes_max = 0;

void
remove_pipe(int fd)
{
  int i, orig_errno = errno;

  for (i = 0; i < pipes_max; i++)
  {
    if (PIPE_FOUND(fd))
    {
      close(fd & ~RW_BIT);
      remove(pipes[i].name);
      free(pipes[i].name);
      errno = orig_errno;
      pipes[i].fds[BACK_UP] = -1;
      pipes[i].name = NULL;
      if (++i == pipes_max)
        pipes_max--;
      return 0;
    }
  }

  errno = orig_errno;
}

int
rpl_close(int fd)
{
  int i;

  for (i = 0; i < pipes_max; i++)
  {
    if (pipes[i].fds[READ_END] == fd)
    {
      /* read end closed.  */
      pipes[i].fds[READ_END] = -1;
      return 0;
    }
    else
    if (pipes[i].fds[WRITE_END] == fd)
    {
      /* write end closed.  */
      int orig_errno = errno;
      if (pipes[i].rewind++)
        pipes[i].fds[WRITE_END] = -1;
      llseek(fd & ~RW_BIT, 0L, SEEK_SET);
      errno = orig_errno;
      return 0;
    }
    else
    if (IS_CLOSED(fd))
    {
      /* both end closed; pipe file removed.  */
      int orig_errno = errno;
      close(fd & ~RW_BIT);
      remove(pipes[i].name);
      free(pipes[i].name);
      errno = orig_errno;
      pipes[i].fds[BACK_UP] = -1;
      pipes[i].name = NULL;
      if (++i == pipes_max)
        pipes_max--;
      return 0;
    }
    else
    if (IS_NOT_CLOSED(fd))
      return 0;  /* one of the ends still open.  */
  }

  return close(fd & ~RW_BIT);
}

int
rpl_pipe(int fds[2])
{
  int fd, i;
  char *tmpdir;
  char *filename;

  if (pipes_max >= OPEN_MAX)
  {
    errno = EMFILE;
    return -1;
  }

  tmpdir = "/dev/env/TMPDIR";
  if (access(tmpdir, D_OK))
  {
    tmpdir = "/dev/env/TMP";
    if (access(tmpdir, D_OK))
    {
      tmpdir = "/dev/env/TEMP";
      if (access(tmpdir, D_OK))
      {
        tmpdir = P_tmpdir;
        if (access(tmpdir, D_OK))
          tmpdir = ".";
      }
    }
  }
  filename = malloc(strlen(tmpdir) + sizeof("/spXXXXXX"));
  if (filename == NULL)
    return -1;
  sprintf(filename, "%s/spXXXXXX", tmpdir);
  filename = mktemp(filename);
  if (filename == NULL)
    return -1;

  fd = open(filename, O_RDWR|O_CREAT|O_TRUNC, 0666);
  if (fd < 0)
  {
    free(filename);
    return -1;
  }
  if (fd < 20)
  {
    int tfd;

    tfd = fcntl(fd, F_DUPFD, 20);
    close(fd);
    if (tfd < 0)
    {
      remove(filename);
      free(filename);
      return -1;
    }
    fd = tfd;
  }

  for (i = 0; i < pipes_max; i++)
    if (pipes[i].name == NULL)
      break;

  pipes[i].rewind = 0;
  pipes[i].name = filename;
  pipes[i].fds[BACK_UP] = fd;
  fds[0] = pipes[i].fds[READ_END] = fd;
  fds[1] = pipes[i].fds[WRITE_END] = fd | RW_BIT;  /* Signals write end of pipe.  */
  if (i == pipes_max && pipes_max < OPEN_MAX)
    pipes_max++;

  return 0;
}

int
rpl_dup(int old_handle)
{
  return dup(old_handle & ~RW_BIT);
}

int
rpl_dup2(int existing_handle, int new_handle)
{
  return dup2(existing_handle & ~RW_BIT, new_handle);
}

ssize_t
rpl_read(int fd, void *buffer, size_t count)
{
  return read(fd & ~RW_BIT, buffer, count);
}

ssize_t
rpl__read(int fd, void *buffer, size_t count)
{
  return _read(fd & ~RW_BIT, buffer, count);
}

int
rpl_write(int fd, const void *buffer, size_t count)
{
  return write(fd & ~RW_BIT, buffer, count);
}

static void
remove_pipe_files(void)
{
  int i;

  for (i = 0; i < pipes_max; i++)
  {
    if (pipes[i].fds[BACK_UP] != -1)
      close(pipes[i].fds[BACK_UP]);
    if (pipes[i].name)
      remove(pipes[i].name);
  }
}

/* This is called before `main' to install the temporary file deleter at exit.  */
static void __attribute__((constructor))
djgpp_bash_startup (void)
{
  atexit(remove_pipe_files);
}

#endif /* __DJGPP__ */
